#include "zpc/helper.hpp"
#include "zpc/printer.hpp"
#include <iostream>

struct Json;
using JsonPtr = std::shared_ptr<Json>;
using JsonV = std::variant<std::string, std::vector<JsonPtr>, std::vector<std::tuple<std::string, JsonPtr>>>;
struct Json {
  Json(JsonV v): v(v) {}
  JsonV v;
};

std::ostream& operator << (std::ostream& os, const Json& json) {
  return os << json.v;
}

int main() {
  auto ws = many(ch([](char c) { return isspace(c); }));
  auto tok = [&](auto&& p) { return seq(p, ws) % RESOLVE_OVERLOAD(std::get<0>); };

  auto sign = alt(lit("+"), lit("-"));
  auto digit = ch_range('0', '9');

  auto json_int = alt(
      lit("0"),
      seq(ch_range('1', '9'), many(digit))
  );
  auto json_exp = seq(
      alt(lit("E"), lit("e")),
      opt(sign),
      json_int
  );
  auto float_part = seq(lit("."), many1(digit));
  Parser<std::string> json_number = tok(raw(seq(
      opt(sign),
      alt(
          seq(json_int, opt(float_part)),
          float_part
      ),
      opt(json_exp)
  )));

  Parser<std::string> json_string = tok(alt(
  seq(lit("\""), raw(many(ch([](char c) { return c != '"'; }))), lit("\"")),
  seq(lit("'"), raw(many(ch([](char c) { return c != '"'; }))), lit("'"))
  )) % RESOLVE_OVERLOAD(std::get<1>);

  Parser<JsonPtr> json_val;

  auto lazy_json_val = lazy(json_val);

  Parser<std::vector<JsonPtr>> json_arr = tok(seq(
      tok(lit("[")),
      sep_by(lazy_json_val, tok(lit(","))),
      tok(lit("]"))
  )) % RESOLVE_OVERLOAD(std::get<1>);

  Parser<std::vector<std::tuple<std::string, JsonPtr>>> json_obj =
      tok(seq(
          tok(lit("{")),
          sep_by(
              seq(
                  json_string, tok(lit(":")), lazy_json_val
              ) %= [](const std::string &k, const std::string &, const JsonPtr &v) { return std::make_tuple(k, v); },
              tok(lit(","))
          ),
          tok(lit("}"))
      )) % RESOLVE_OVERLOAD(std::get<1>);

  Parser<JsonV> _json_val = tok(alt(
      json_number,
      json_string,
      tok(lit("true")),
      tok(lit("false")),
      tok(lit("null")),
      json_arr,
      json_obj
  ));
  json_val = _json_val % [](const JsonV &v) { return std::make_shared<Json>(v); };

  auto json = eof(json_obj);

  std::string_view in = R"|({
    "a": [1, 2, 3],
    "b": 1,
    "c": {
      "x": 1.1,
      "y":.1e2,
      "z": -2.e+2
    }
  })|";
  auto res = json(in);
  std::cout << res << std::endl;
  std::cout << "Remain: " << in << std::endl;
}