#ifndef ZPC_PARSER_HPP
#define ZPC_PARSER_HPP

#include <string_view>
#include <functional>
#include <string>

#include "unique_variant.hpp"
#include "optional.hpp"

template<typename T>
using ParseResult = optional<T>;

template<typename T>
using Parser = std::function<ParseResult<T>(std::string_view&)>;


template<typename F, typename T, typename R = std::remove_reference_t<std::invoke_result_t<F, T>>>
Parser<R> operator % (const Parser<T>& p, F&& f) {
  return [=](std::string_view& in)->ParseResult<R> {
    return p(in).map([&](auto&& v) { return f(v); });
  };
}

template<typename F, typename... Ts, typename R = std::remove_reference_t<std::invoke_result_t<F, Ts...>>>
Parser<R> operator %= (const Parser<std::tuple<Ts...>>& p, F&& f) {
  return [=](std::string_view& in)->ParseResult<R> {
    return p(in).map([&](auto&& v) { return std::apply(f, v); });
  };
}

#define RESOLVE_OVERLOAD(...) \
	[](auto&&...args)->decltype(auto){return __VA_ARGS__(std::forward<decltype(args)>(args)...);}


Parser<std::tuple<>> seq() {
  return [](std::string_view&)->ParseResult<std::tuple<>> {
    return std::make_tuple();
  };
}

template<typename T, typename... Ts, typename R = std::tuple<T, Ts...>>
Parser<R> seq(const Parser<T>& p, const Parser<Ts>& ...ps) {
  return [=](std::string_view& in)->ParseResult<R> {
    return p(in).flat_map([&](auto v) {
      return seq(ps...)(in).map([&](auto vs) {
        return std::tuple_cat(std::make_tuple(v), vs);
      });
    });
  };
}

template<typename T>
Parser<T> attempt(const Parser<T>& p) {
  return [=](std::string_view& in)->ParseResult<T> {
    auto in_bak = in;
    auto v = p(in);
    if (!v) in = in_bak;
    return v;
  };
}

namespace detail {
  template<typename R>
  Parser<R> alt() {
    return [](std::string_view&)->ParseResult<R> {
      return {};
    };
  }

  template<typename R, typename T, typename... Ts>
  Parser<R> alt(const Parser<T>& p, const Parser<Ts>& ...ps) {
    return [=](std::string_view& in)->ParseResult<R> {
      return attempt(p)(in).map([](auto v){ return R{v}; })
          .or_else(alt<R>(ps...)(in));
    };
  }

  template<typename... Ts>
  auto flat(const std::variant<Ts...>& v) {
    if constexpr (sizeof...(Ts) == 1) {
      return std::get<0>(v);
    } else {
      return v;
    }
  }
}

template<typename... Ts>
auto alt(const Parser<Ts>& ...ps) {
  return detail::alt<unique_variant_t<Ts...>>(ps...) % RESOLVE_OVERLOAD(detail::flat);
}

template<typename T, typename R = std::vector<T>>
Parser<R> many(const Parser<T>& p) {
  return [=](std::string_view& in)->ParseResult<R> {
    R r;
    ParseResult<T> v;
    while ((v = attempt(p)(in))) {
      r.emplace_back(v.value());
    }
    return r;
  };
}

template<typename T>
Parser<T> lazy(const Parser<T>& p) {
  return [p_addr = &p](std::string_view& in)->ParseResult<T> {
    return (*p_addr)(in);
  };
}

template<typename T, typename R = optional<T>>
Parser<R> opt(const Parser<T>& p) {
  return [=](std::string_view& in)->ParseResult<R> {
    return p(in);
  };
}

#endif //ZPC_PARSER_HPP
